<?php

namespace Oro\Bundle\DataGridBundle\Datagrid;

class ParameterBag
{
    const ADDITIONAL_PARAMETERS = '_parameters';
    const MINIFIED_PARAMETERS   = '_minified';

    /**
     * Parameter storage.
     *
     * @var array
     */
    protected $parameters;

    /**
     * Constructor.
     *
     * @param array $parameters An array of parameters
     */
    public function __construct(array $parameters = array())
    {
        $this->parameters = $parameters;
    }

    /**
     * Returns the parameters.
     *
     * @return array An array of parameters
     */
    public function all()
    {
        return $this->parameters;
    }

    /**
     * Returns the parameter keys.
     *
     * @return array An array of parameter keys
     */
    public function keys()
    {
        return array_keys($this->parameters);
    }

    /**
     * Replaces the current parameters by a new set.
     *
     * @param array $parameters An array of parameters
     */
    public function replace(array $parameters = array())
    {
        $this->parameters = $parameters;
    }

    /**
     * Adds parameters.
     *
     * @param array $parameters An array of parameters
     */
    public function add(array $parameters = array())
    {
        $this->parameters = array_replace_recursive($this->parameters, $parameters);
    }

    /**
     * Get parameter value
     *
     * @param string $key
     * @param mixed  $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        return $this->has($key) ? $this->parameters[$key] : $default;
    }

    /**
     * Sets a parameter by key.
     *
     * @param string $key   The key
     * @param mixed  $value The value
     */
    public function set($key, $value)
    {
        $this->parameters[$key] = $value;
    }

    /**
     * Merge parameter value by key
     *
     * @param string $key
     * @param mixed  $value
     */
    public function mergeKey($key, $value)
    {
        $currentValue = $this->get($key);

        if (is_array($currentValue) && is_array($value)) {
            $value = array_replace_recursive($currentValue, $value);
        }

        $this->set($key, $value);
    }

    /**
     * Returns true if the parameter is defined.
     *
     * @param string $key The key
     * @return Boolean true if the parameter exists, false otherwise
     */
    public function has($key)
    {
        return array_key_exists($key, $this->parameters);
    }

    /**
     * Removes a parameter.
     *
     * @param string $key The key
     */
    public function remove($key)
    {
        unset($this->parameters[$key]);
    }
}
